from django.db import models
from django.utils import timezone

class Category(models.Model):
    name = models.CharField(max_length=100)
    slug = models.SlugField(unique=True)
    description = models.TextField(blank=True)
    
    class Meta:
        verbose_name_plural = "Categories"
    
    def __str__(self):
        return self.name

class Product(models.Model):
    LIQUID = 'liquid'
    POWDER = 'powder'
    PODS = 'pods'
    BAR = 'bar'
    
    FORM_CHOICES = [
        (LIQUID, 'Liquid'),
        (POWDER, 'Powder'),
        (PODS, 'Pods/Capsules'),
        (BAR, 'Bar'),
    ]
    
    name = models.CharField(max_length=200)
    image = models.ImageField(
        upload_to='products/%Y/%m/%d/',
        blank=True,
        null=True,
        help_text="Upload product image (optional)"
    )
    image_url = models.URLField(
        blank=True,
        null=True,
        default='https://images.unsplash.com/photo-1600857062241-98c2aae9d15f?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80',
        help_text="URL for product image (optional - will be used if no image uploaded)"
    )
    slug = models.SlugField(unique=True)
    description = models.TextField()
    category = models.ForeignKey(Category, related_name='products', on_delete=models.CASCADE)
    form = models.CharField(max_length=20, choices=FORM_CHOICES, default=LIQUID)
    volume = models.CharField(max_length=50, help_text="e.g., 1kg, 800gms, 600gms, 200gms")
    ingredients = models.TextField(blank=True)
    usage_instructions = models.TextField(blank=True)
    in_stock = models.BooleanField(default=True)
    is_active = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now=True)
    
    def __str__(self):
        return self.name
    
    @property
    def display_image(self):
        """
        Return the image to display - prefers uploaded image, falls back to URL,
        and finally to a default image if neither is available.
        """
        if self.image:
            return self.image.url
        elif self.image_url:
            return self.image_url
        else:
            return 'https://images.unsplash.com/photo-1600857062241-98c2aae9d15f?ixlib=rb-4.0.3&auto=format&fit=crop&w=500&q=80'
    
    @property
    def has_image(self):
        """Check if the product has any image available"""
        return bool(self.image or self.image_url)
    
    class Meta:
        ordering = ['-created']